/**
 * @license
 * Copyright 2021 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
// Workaround for allowing cjs module to be included in bundle created by
// rollup.
import * as LongExports from 'long';
// tslint:disable-next-line
const Long = 
// tslint:disable-next-line
LongExports.default || LongExports;
export function hexToLong(hex) {
    return Long.fromString(hex, true, 16);
}
// Some primes between 2^63 and 2^64 for various uses.
// Hex 0xc3a5c85c97cb3127
const k0 = hexToLong('c3a5c85c97cb3127');
// Hex 0xb492b66fbe98f273
const k1 = hexToLong('b492b66fbe98f273');
// Hex 0x9ae16a3b2f90404f
const k2 = hexToLong('9ae16a3b2f90404f');
function shiftMix(val) {
    return val.xor(val.shru(47));
}
function fetch(s, offset, numBytes) {
    const bytes = s.slice(offset, offset + numBytes);
    return Long.fromBytes(Array.from(bytes), true, true);
}
function fetch64(s, offset) {
    return fetch(s, offset, 8);
}
function fetch32(s, offset) {
    return fetch(s, offset, 4);
}
function rotate64(val, shift) {
    // Avoid shifting by 64: doing so yields an undefined result.
    return shift === 0 ? val : val.shru(shift).or(val.shl(64 - shift));
}
function hashLen16(u, v, mul = hexToLong('9ddfea08eb382d69')) {
    // Murmur-inspired hashing.
    let a = u.xor(v).mul(mul);
    a = a.xor(a.shru(47));
    let b = v.xor(a).mul(mul);
    b = b.xor(b.shru(47));
    b = b.mul(mul);
    return b;
}
// Return a 16-byte hash for 48 bytes.  Quick and dirty.
// Callers do best to use "random-looking" values for a and b.
function weakHashLen32WithSeeds(w, x, y, z, a, b) {
    a = a.add(w);
    b = rotate64(b.add(a).add(z), 21);
    const c = a;
    a = a.add(x);
    a = a.add(y);
    b = b.add(rotate64(a, 44));
    return [a.add(z), b.add(c)];
}
function weakHashLen32WithSeedsStr(s, offset, a, b) {
    return weakHashLen32WithSeeds(fetch64(s, offset), fetch64(s, offset + 8), fetch64(s, offset + 16), fetch64(s, offset + 24), a, b);
}
function hashLen0to16(s, len = s.length) {
    if (len >= 8) {
        const mul = k2.add(len * 2);
        const a = fetch64(s, 0).add(k2);
        const b = fetch64(s, len - 8);
        const c = rotate64(b, 37).mul(mul).add(a);
        const d = rotate64(a, 25).add(b).mul(mul);
        return hashLen16(c, d, mul);
    }
    if (len >= 4) {
        const mul = k2.add(len * 2);
        const a = fetch32(s, 0);
        return hashLen16(a.shl(3).add(len), fetch32(s, len - 4), mul);
    }
    if (len > 0) {
        const a = s[0];
        const b = s[len >> 1];
        const c = s[len - 1];
        const y = a + (b << 8);
        const z = len + (c << 2);
        return shiftMix(k2.mul(y).xor(k0.mul(z))).mul(k2);
    }
    return k2;
}
function hashLen17to32(s, len = s.length) {
    const mul = k2.add(len * 2);
    const a = fetch64(s, 0).mul(k1);
    const b = fetch64(s, 8);
    const c = fetch64(s, len - 8).mul(mul);
    const d = fetch64(s, len - 16).mul(k2);
    return hashLen16(rotate64(a.add(b), 43).add(rotate64(c, 30)).add(d), a.add(rotate64(b.add(k2), 18)).add(c), mul);
}
function hashLen33to64(s, len = s.length) {
    const mul = k2.add(len * 2);
    const a = fetch64(s, 0).mul(k2);
    const b = fetch64(s, 8);
    const c = fetch64(s, len - 8).mul(mul);
    const d = fetch64(s, len - 16).mul(k2);
    const y = rotate64(a.add(b), 43).add(rotate64(c, 30)).add(d);
    const z = hashLen16(y, a.add(rotate64(b.add(k2), 18)).add(c), mul);
    const e = fetch64(s, 16).mul(mul);
    const f = fetch64(s, 24);
    const g = y.add(fetch64(s, len - 32)).mul(mul);
    const h = z.add(fetch64(s, len - 24)).mul(mul);
    return hashLen16(rotate64(e.add(f), 43).add(rotate64(g, 30)).add(h), e.add(rotate64(f.add(a), 18)).add(g), mul);
}
export function fingerPrint64(s, len = s.length) {
    const seed = Long.fromNumber(81, true);
    if (len <= 32) {
        if (len <= 16) {
            return hashLen0to16(s, len);
        }
        else {
            return hashLen17to32(s, len);
        }
    }
    else if (len <= 64) {
        return hashLen33to64(s, len);
    }
    // For strings over 64 bytes we loop.  Internal state consists of
    // 56 bytes: v, w, x, y, and z.
    let x = seed;
    let y = seed.mul(k1).add(113);
    let z = shiftMix(y.mul(k2).add(113)).mul(k2);
    let v = [Long.UZERO, Long.UZERO];
    let w = [Long.UZERO, Long.UZERO];
    x = x.mul(k2).add(fetch64(s, 0));
    let offset = 0;
    // Set end so that after the loop we have 1 to 64 bytes left to process.
    const end = ((len - 1) >> 6) * 64;
    const last64 = end + ((len - 1) & 63) - 63;
    do {
        x = rotate64(x.add(y).add(v[0]).add(fetch64(s, offset + 8)), 37).mul(k1);
        y = rotate64(y.add(v[1]).add(fetch64(s, offset + 48)), 42).mul(k1);
        x = x.xor(w[1]);
        y = y.add(v[0]).add(fetch64(s, offset + 40));
        z = rotate64(z.add(w[0]), 33).mul(k1);
        v = weakHashLen32WithSeedsStr(s, offset, v[1].mul(k1), x.add(w[0]));
        w = weakHashLen32WithSeedsStr(s, offset + 32, z.add(w[1]), y.add(fetch64(s, offset + 16)));
        [z, x] = [x, z];
        offset += 64;
    } while (offset !== end);
    const mul = k1.add(z.and(0xff).shl(1));
    // Point to the last 64 bytes of input.
    offset = last64;
    w[0] = w[0].add((len - 1) & 63);
    v[0] = v[0].add(w[0]);
    w[0] = w[0].add(v[0]);
    x = rotate64(x.add(y).add(v[0]).add(fetch64(s, offset + 8)), 37).mul(mul);
    y = rotate64(y.add(v[1]).add(fetch64(s, offset + 48)), 42).mul(mul);
    x = x.xor(w[1].mul(9));
    y = y.add(v[0].mul(9).add(fetch64(s, offset + 40)));
    z = rotate64(z.add(w[0]), 33).mul(mul);
    v = weakHashLen32WithSeedsStr(s, offset, v[1].mul(mul), x.add(w[0]));
    w = weakHashLen32WithSeedsStr(s, offset + 32, z.add(w[1]), y.add(fetch64(s, offset + 16)));
    [z, x] = [x, z];
    return hashLen16(hashLen16(v[0], w[0], mul).add(shiftMix(y).mul(k0)).add(z), hashLen16(v[1], w[1], mul).add(x), mul);
}
//# sourceMappingURL=data:application/json;base64,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